const express = require('express');
const Class = require('../../models/Class');
const Teacher = require('../../models/teacher/Teacher');
const Student = require('../../models/student/Student');
const { teacherAuth } = require('../../middleware/auth');

const router = express.Router();

router.get('/dashboard', teacherAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        teacher: {
          name: 'Development Teacher',
          email: 'teacher@edumetrix.uk',
          subjects: ['Mathematics', 'Physics'],
          assignedStudentsCount: 15
        },
        todaysClasses: [
          {
            _id: 'class1',
            subject: 'Mathematics',
            scheduledDate: new Date(),
            duration: 60,
            student: { studentName: 'John Doe' },
            status: 'scheduled'
          },
          {
            _id: 'class2', 
            subject: 'Physics',
            scheduledDate: new Date(Date.now() + 2 * 60 * 60 * 1000),
            duration: 90,
            student: { studentName: 'Jane Smith' },
            status: 'scheduled'
          }
        ],
        upcomingClasses: [
          {
            _id: 'class3',
            subject: 'Mathematics',
            scheduledDate: new Date(Date.now() + 24 * 60 * 60 * 1000),
            duration: 60,
            student: { studentName: 'Mike Wilson' }
          }
        ],
        monthlyStats: {
          classes: 28,
          earnings: 2800,
          hours: 45,
          studentsActive: 15
        }
      });
    }
    
    const teacher = await Teacher.findOne({ userId: req.user._id })
      .populate('assignedStudents.studentId', 'studentName');

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher profile not found' });
    }

    const today = new Date();
    const startOfDay = new Date(today.setHours(0, 0, 0, 0));
    const endOfDay = new Date(today.setHours(23, 59, 59, 999));

    const todaysClasses = await Class.find({
      teacherId: teacher._id,
      scheduledDate: { $gte: startOfDay, $lte: endOfDay },
      status: { $in: ['scheduled', 'ongoing'] }
    })
    .populate('studentId', 'studentName')
    .sort({ scheduledDate: 1 });

    const upcomingClasses = await Class.find({
      teacherId: teacher._id,
      scheduledDate: { $gt: endOfDay },
      status: 'scheduled'
    })
    .populate('studentId', 'studentName')
    .sort({ scheduledDate: 1 })
    .limit(5);

    const currentMonth = new Date();
    const startOfMonth = new Date(currentMonth.getFullYear(), currentMonth.getMonth(), 1);
    const endOfMonth = new Date(currentMonth.getFullYear(), currentMonth.getMonth() + 1, 0);

    const monthlyClasses = await Class.countDocuments({
      teacherId: teacher._id,
      scheduledDate: { $gte: startOfMonth, $lte: endOfMonth },
      status: 'completed'
    });

    let monthlyEarnings = 0;
    if (teacher.salaryType === 'hourly') {
      const completedClasses = await Class.find({
        teacherId: teacher._id,
        scheduledDate: { $gte: startOfMonth, $lte: endOfMonth },
        status: 'completed'
      });
      
      const totalHours = completedClasses.reduce((sum, cls) => sum + (cls.duration / 60), 0);
      monthlyEarnings = totalHours * teacher.salary.amount;
    } else {
      monthlyEarnings = teacher.salary.amount;
    }

    res.json({
      teacher: {
        name: teacher.name,
        subjects: teacher.subjects,
        salaryType: teacher.salaryType,
        totalStudents: teacher.assignedStudents.length
      },
      todaysClasses,
      upcomingClasses,
      stats: {
        monthlyClasses,
        monthlyEarnings: monthlyEarnings.toFixed(2),
        totalStudents: teacher.assignedStudents.length
      }
    });
  } catch (error) {
    console.error('Teacher dashboard error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/classes', teacherAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findOne({ userId: req.user._id });
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher profile not found' });
    }

    const { page = 1, limit = 10, status = 'all', date } = req.query;
    
    let query = { teacherId: teacher._id };
    
    if (status !== 'all') {
      query.status = status;
    }
    
    if (date) {
      const startDate = new Date(date);
      const endDate = new Date(date);
      endDate.setDate(endDate.getDate() + 1);
      query.scheduledDate = { $gte: startDate, $lt: endDate };
    }

    const classes = await Class.find(query)
      .populate('studentId', 'studentName')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ scheduledDate: -1 });

    const total = await Class.countDocuments(query);

    res.json({
      classes,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get teacher classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/students', teacherAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findOne({ userId: req.user._id })
      .populate({
        path: 'assignedStudents.studentId',
        select: 'studentName email subjects class country'
      });

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher profile not found' });
    }

    const studentsWithSubjects = teacher.assignedStudents.map(assignment => ({
      student: assignment.studentId,
      subject: assignment.subject,
      canChat: true
    }));

    res.json({ students: studentsWithSubjects });
  } catch (error) {
    console.error('Get teacher students error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/classes/:id/start', teacherAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findOne({ userId: req.user._id });
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher profile not found' });
    }

    const classDoc = await Class.findOne({
      _id: req.params.id,
      teacherId: teacher._id
    });

    if (!classDoc) {
      return res.status(404).json({ message: 'Class not found' });
    }

    classDoc.status = 'ongoing';
    await classDoc.save();

    res.json({ message: 'Class started successfully', class: classDoc });
  } catch (error) {
    console.error('Start class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/classes/:id/complete', teacherAuth, async (req, res) => {
  try {
    const { studentAttended, classNotes, homeworkAssigned } = req.body;
    
    const teacher = await Teacher.findOne({ userId: req.user._id });
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher profile not found' });
    }

    const classDoc = await Class.findOne({
      _id: req.params.id,
      teacherId: teacher._id
    });

    if (!classDoc) {
      return res.status(404).json({ message: 'Class not found' });
    }

    classDoc.status = 'completed';
    classDoc.attendanceStatus = {
      studentAttended: studentAttended === true,
      teacherAttended: true
    };
    classDoc.classNotes = classNotes;
    classDoc.homeworkAssigned = homeworkAssigned;

    await classDoc.save();

    res.json({ message: 'Class completed successfully', class: classDoc });
  } catch (error) {
    console.error('Complete class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/profile', teacherAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findOne({ userId: req.user._id })
      .populate('userId', 'username');

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher profile not found' });
    }

    res.json({ teacher });
  } catch (error) {
    console.error('Get teacher profile error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/monthly-report', teacherAuth, async (req, res) => {
  try {
    const { month, year } = req.query;
    
    const teacher = await Teacher.findOne({ userId: req.user._id });
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher profile not found' });
    }

    const currentDate = new Date();
    const reportMonth = month || (currentDate.getMonth() + 1);
    const reportYear = year || currentDate.getFullYear();

    const startDate = new Date(reportYear, reportMonth - 1, 1);
    const endDate = new Date(reportYear, reportMonth, 0);

    const classes = await Class.find({
      teacherId: teacher._id,
      scheduledDate: { $gte: startDate, $lte: endDate },
      status: 'completed',
      'attendanceStatus.teacherAttended': true
    })
    .populate('studentId', 'studentName')
    .sort({ scheduledDate: 1 });

    const totalClasses = classes.length;
    const totalHours = classes.reduce((sum, cls) => sum + (cls.duration / 60), 0);

    let earnings = 0;
    if (teacher.salaryType === 'hourly') {
      earnings = totalHours * teacher.salary.amount;
    } else {
      earnings = teacher.salary.amount;
    }

    const classDetails = classes.map(cls => ({
      date: cls.scheduledDate.toISOString().split('T')[0],
      student: cls.studentId.studentName,
      subject: cls.subject,
      duration: cls.duration,
      notes: cls.classNotes
    }));

    res.json({
      month: `${reportMonth}/${reportYear}`,
      summary: {
        totalClasses,
        totalHours: totalHours.toFixed(2),
        earnings: earnings.toFixed(2),
        salaryType: teacher.salaryType
      },
      classes: classDetails
    });
  } catch (error) {
    console.error('Get teacher monthly report error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;